package models

import (
	"encoding/json"
	"fmt"
	_ "github.com/jackc/pgx/stdlib"
	"github.com/jmoiron/sqlx"
	"io/ioutil"
	"log"
	"os"
)

type config struct {
	Host    string `json:"host"`
	Port    int    `json:"port"`
	SphPort int    `json:"sphPort"`
	User    string `json:"user"`
	Pass    string `json:"pass"`
	DbName  string `json:"dbName"`
}

var db *sqlx.DB

func InitDb() {
	cfg := loadConfig()
	format := "host=%s port=%d user=%s password=%s dbname=%s"
	db = dbConnect("pgx", fmt.Sprintf(format, cfg.Host, cfg.Port, cfg.User, cfg.Pass, cfg.DbName))
	db.SetMaxOpenConns(5)
	db.SetMaxIdleConns(5)
}

func loadConfig() config {
	hostname, err := os.Hostname()
	if err != nil {
		log.Fatalln(err)
	}

	jsonFile, err := os.Open( getPathConfigFile(hostname))
	if err != nil {
		log.Fatalln(err)
	}
	defer jsonFile.Close()

	byteValue, err := ioutil.ReadAll(jsonFile)
	if err != nil{
		log.Fatalln(err)
	}

	cfg := config{}
	err = json.Unmarshal(byteValue, &cfg)
	if err != nil{
		log.Fatalln(err)
	}
	return cfg
}

func dbConnect(driver string, config string) *sqlx.DB {
	//psqlInfo := fmt.Sprintf("host=%s port=%d user=%s password=%s dbname=%s", host, port, user, password, dbname)
	//db, err := sqlx.Connect("pgx", "postgres://storage:OiNMlrkxE3fhD0Xs@localhost:5432/storage")
	db, err := sqlx.Connect(driver, config)
	if err != nil {
		log.Fatalln(err)
	}

	return db
}

func rootDir() string{
	path, err := os.Getwd()
	if err != nil{
		log.Fatalln(err)
	}
	return path
}

func checkFileExists(filepath string) string{
	_, err := os.Stat(filepath)
	if os.IsNotExist(err) {
		log.Fatalln("File not exists: "+ filepath)
	}

	return filepath
}

func getPathConfigFile(name string) string{
	filepath := rootDir()+"/configs/"+name+".json"
	return checkFileExists(filepath)
}